VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
  Persistable = 0  'NotPersistable
  DataBindingBehavior = 0  'vbNone
  DataSourceBehavior  = 0  'vbNone
  MTSTransactionMode  = 0  'NotAnMTSObject
END
Attribute VB_Name = "Utils"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = True
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = True
'**********************************************************
' Class SoapUtils.Utils
' Some funtions to encode method calls into SOAP transaction requests
'
' Errors are handled in a few ways:
' SOAP infrastructure errors are stored in the SOAPFault*
' properties, and are returned as part of the response body
' in the SOAP:Fault element if there is a SOAP-level error.
'
' Back-end application infrastructure errors (those raised by
' the back-end code that is actually doing the request work) are
' stored in the Detail struct, and may be returned as part of the
' SOAP:Fault element
'
' Finally, errors raised in this code directly (COM errors,
' objects not set, etc.) are stored in the properties of
' ClassErrors, an instance of ErrorUtils.ErrorInfo.  This
' class exposes properties for retrieving the error info as well
' as getting the error info as an XML string.
'**********************************************************
Option Explicit


Const DICT_PROGID_SUFFIX As String = "PROGID"
Const DICT_METHOD_SUFFIX As String = "METHOD"
Const DICT_SERVER_SUFFIX As String = "SERVER"
Const PROGID_STRING As String = "SoapUtils.Utils"
Private m_dictBindings As Scripting.Dictionary

Public ClassErrors As ErrorUtils.ErrorInfo

Type udtErrorDetail
    Namespace As String     ' namespace prefix for error detail
    NamespaceURI As String  ' URI of error namespace
    Datatype As String      ' error datatype
    Errorcode As Long       ' error code for error
    Description As String   ' Some useful info
    Location As String      ' Where it happened
End Type

Private Detail As udtErrorDetail

Enum SOAP_FAULT_CODE
    Version_Mismatch = 100
    Must_Understand = 200
    Invalid_Request = 300
    Application_Faulted = 400
End Enum

Enum SOAP_RUN_CODE
    Maybe = 0
    No = 1
    Yes = 2
End Enum

Private m_sSoapFaultDescription As String
Private m_sSoapFaultString As String
Private m_nSoapFaultCode As SOAP_FAULT_CODE
Private m_nSoapRunCode As SOAP_RUN_CODE

Public Property Get FunctionErrorDescription() As String
    FunctionErrorDescription = ClassErrors.Description
End Property
Public Property Get FunctionErrorNumber() As Long
    FunctionErrorNumber = ClassErrors.Number
End Property
Public Property Get FunctionErrorLocation() As String
    FunctionErrorLocation = ClassErrors.Location
End Property

'**********************************************************
' Public Function BuildDetailXml() As String
' Use the detail structure to build an XML fragment to
' return with the SOAP:Fault element
'**********************************************************
Public Function BuildDetailXml() As String
    On Error GoTo ErrHand
    Dim sXML As String
    Dim sErrNsAttr As String
    Dim sXsdNsAttr As String
    
    If Not DetailsAreSet() Then
        sXML = ""
    Else
        sXML = ""
        sXsdNsAttr = "xmlns:xsd='www.w3.org/schemas'"
        With Detail
            sErrNsAttr = "xmlns:" & .Namespace & "='" & .NamespaceURI & _
                    "' xsd:type='" & .Namespace & ":" & .Datatype & "' "
            sXML = sXML & "<detail " & sXsdNsAttr & " " & _
                sErrNsAttr & " >" & Chr(10)
            sXML = sXML & "<message>" & .Description & "</message>" & Chr(10)
            sXML = sXML & "<location>" & .Location & "</location>" & Chr(10)
            sXML = sXML & "<errorcode>" & CStr(.Errorcode) & _
                "</errorcode>" & Chr(10)
            sXML = sXML & "</detail>"
        End With
    End If

ErrHand:
    If Err.Number <> 0 Then
        Call ClassErrors.SetErrorInfo(Err.Description, _
                PROGID_STRING & ".BuildDetailXml", _
                Err.Number)
        sXML = ""
    End If
    
    BuildDetailXml = sXML
End Function


'**********************************************************
' Public Function BuildFaultXml() As String
' If there's an error in the SOAP infrastructure then we need to
' return a SOAP:Fault element.  Fault has a few child nodes to
' hold error info; there's also the option of including your
' own error details, which requires a whole other set of XML.
' The function uses BuildDetailXML to create this if IncludeDetail
' is True.
'**********************************************************
Public Function BuildFaultXml(Optional bIncludeDetail As Boolean = False) As String
    On Error GoTo ErrHand
    
    Dim sXML As String
    sXML = ""
    
    sXML = sXML & "<SOAP:Fault>" & Chr(10)
    sXML = sXML & "<faultcode>" & m_nSoapFaultCode & "</faultcode>" & Chr(10)
    sXML = sXML & "<faultstring>" & m_sSoapFaultString & "</faultstring>" & Chr(10)
    sXML = sXML & "<runcode>" & m_nSoapRunCode & "</runcode>" & Chr(10)
    If (bIncludeDetail) Then

        sXML = sXML & BuildDetailXml() & Chr(10)
    End If
    sXML = sXML & "</SOAP:Fault>"
    
ErrHand:
    If Err.Number <> 0 Then
        Call ClassErrors.SetErrorInfo(Err.Description, _
                PROGID_STRING & ".BuildFaultXml", _
                Err.Number)
        sXML = ""
    End If
    BuildFaultXml = sXML
End Function
'**********************************************************
' Private Function BuildSoapEnvelopeFault(bIncludeDetail As Boolean) As String
' Grab the Fault XML and wrap the correct elements around it
'**********************************************************
Private Function BuildSoapEnvelopeFault(bIncludeDetail As Boolean) As String
    Dim sRes As String
    sRes = "<SOAP:Envelope xmlns:SOAP='urn:schemas-xmlsoap-org:soap.v1' " & Chr(10)
    sRes = sRes & " SOAP:encodingStyle='urn:schemas-xmlsoap-org:soap.v1'>" & Chr(10)
    sRes = sRes & " <SOAP:Body>" & Chr(10)
    sRes = sRes & BuildFaultXml(bIncludeDetail)
    sRes = sRes & "</SOAP:Body>" & Chr(10)
    sRes = sRes & "</SOAP:Envelope>"
    
    BuildSoapEnvelopeFault = sRes

End Function
'**********************************************************
' Private Function DetailsAreSet() As Boolean
' Method to verify that the required Detail info is
' set to something.
'**********************************************************
Private Function DetailsAreSet() As Boolean
    With Detail
        If Len(.Datatype) < 1 Then
            DetailsAreSet = False
        ElseIf (Len(.Namespace) < 1) Then
            DetailsAreSet = False
        ElseIf (Len(.Description) < 1) Then
            DetailsAreSet = False
        ElseIf (Len(.NamespaceURI) < 1) Then
            DetailsAreSet = False
        ElseIf (Len(.Location) < 1) Then
            DetailsAreSet = False
        Else
            DetailsAreSet = True
        End If
    End With
    
End Function

Public Function EncodeParameter(sParamName As String, vParamValue As Variant) As String
    On Error GoTo ErrHand
'Type Example
'integer  58502
'Real 3.14159265358979E+15
'negative-integer  -32768
' http://www.w3.org/1999/XMLSchema
' xmlns:xsd="http://www.w3.org/1999/XMLSchema"
' xsd:type="some type"
    Dim sParamType As String
    Dim sEncType
    Dim sName As String
    Dim sEnc As String
    Dim bIsArray As Boolean
    Dim sType As String
    
    sType = TypeName(vParamValue)
    If InStr(sType, "()") Then
        bIsArray = True
        sType = Trim(Replace(sType, "()", ""))
    End If
    
    Select Case sType
        Case "Boolean"
            sEncType = "boolean"
        Case "Byte"
        
        Case "Currency"
        Case "Date"
        Case "Decimal"
        Case "Double"
        Case "Empty"
        Case "Error"
        Case "Integer"
        Case "Long"
        Case "Nothing"
        Case "Null"
        Case "Object"
        Case "Single"
        Case "Unknown"
        Case "Variant()"
        Case Else
    End Select
ErrHand:
    If Err.Number <> 0 Then
        Call ClassErrors.SetErrorInfo(Err.Description, PROGID_STRING & ".EncodeParameter", Err.Number)
    End If

End Function

'**********************************************************
' Public Function ExecuteRequestWrapper(sSoapXML As String) As String
' Takes the SOAP request, decodes the method and params, loads
' the corresponding VB object wrapper, and calls it. Results as are
' encoded as a SOAP response.
' Note that CalByName requires that we know the number of parameters
' in advance. Hmm... To be able to call *any* object, we need to have
' the destination method wrapped in a method that takes a single String
' parameter: the SOAP parameer XML. The wrapper must figure out how to call
' some other method if need be.
'**********************************************************
Public Function ExecuteRequestWrapper(sSOAPXML As String) As String
    On Error GoTo ErrHand
    
    Dim objX As Object
    Dim sProgID As String
    Dim sSoapMethodName As String
    Dim sComMethodName As String
    Dim sSoapParamXML As String
    Dim sObjectServer As String
    Dim sResults As String
    Dim sHttpHeader As String
    
    ' Check that we can handle the SOAP version
    If VerifySoapVersion(sSOAPXML) Then
        ' We need to parse out the name of the requested method, use
        ' that to index the ProgID of the wrapper in the bindings dictionary,
        ' create the wrapper, and pass in the SOAP paramter string
        sSoapMethodName = RetrieveSoapMethodName(sSOAPXML)
        If Len(sSoapMethodName) Then
            sProgID = m_dictBindings(sSoapMethodName & DICT_PROGID_SUFFIX)
            sComMethodName = m_dictBindings(sSoapMethodName & DICT_METHOD_SUFFIX)
            sObjectServer = m_dictBindings(sSoapMethodName & DICT_SERVER_SUFFIX)
            sSoapParamXML = sSOAPXML
            If Len(sProgID) < 1 Then
            ' No entry in the Dictionary
                Call SetSoapFault(Invalid_Request, No)
                Call SetDetailInfo("MyFault", _
                                    "e", _
                                    "Method not supported", _
                                    "www.wrox.org/VbSoap", _
                                    "ExecuteRequestWrapper", _
                                    -1)
                sResults = BuildSoapEnvelopeFault(True)
            Else
                If Len(sObjectServer) Then
                    Set objX = CreateObject(sProgID, sObjectServer)
                Else
                    Set objX = CreateObject(sProgID)
                End If
                
                If Not objX Is Nothing Then
                    sResults = "<SOAP:Envelope xmlns:SOAP='urn:schemas-xmlsoap-org:soap.v1' " & Chr(10)
                    sResults = sResults & " SOAP:encodingStyle = 'urn:schemas-xmlsoap-org:soap.v1'>" & Chr(10)
                    sResults = sResults & "<SOAP:Body>" & Chr(10)
                    sResults = sResults & CallByName(objX, sComMethodName, VbMethod, sSoapParamXML) & Chr(10)
                    sResults = sResults & "</SOAP:Body>" & Chr(10)
                    sResults = sResults & "</SOAP:Envelope>"
                    Set objX = Nothing
                Else
                    ' Failed to create the object
                    Call SetSoapFault(Application_Faulted, No)
                    Call SetDetailInfo("MyFault", _
                                    "e", _
                                    "Failed to create object", _
                                    "www.wrox.org/VbSoap", _
                                    "ExecuteRequestWrapper", _
                                    -1)
                    sResults = BuildSoapEnvelopeFault(True)
                End If
            End If
        End If
    Else
        ' Wrong version!
        Call SetSoapFault(Version_Mismatch, No)
        sResults = BuildSoapEnvelopeFault(True)
    End If
    
ErrHand:
    If Err.Number <> 0 Then
        Call SetDetailInfo("MyFault", _
                "e", _
                "COM error" & Err.Description, _
                "www.wrox.org/VbSoap", _
                "ExecuteRequestWrapper", _
                Err.Number)
        Call SetSoapFault(Application_Faulted, No)
        
        sResults = BuildSoapEnvelopeFault(True)
        Call ClassErrors.SetErrorInfo(Err.Description, _
                PROGID_STRING & ".ExecuteRequestWrapper ", _
                Err.Number)
    End If
    
    sHttpHeader = "HTTP/1.1 200 OK" & Chr(10)
    sHttpHeader = sHttpHeader & "Content-Type: text/xml" & Chr(10)
    sHttpHeader = sHttpHeader & "Content -length: "
    sHttpHeader = sHttpHeader & CStr(Len(sResults)) & Chr(10) & Chr(10)
    
    ExecuteRequestWrapper = sHttpHeader & sResults
End Function


Public Function GetErrorInfoXML() As String
    GetErrorInfoXML = ClassErrors.GetErrorInfoXML()
End Function

'**********************************************************
' Public Function LoadMethodBindings() As Boolean
' Reads in an XML file that defines what objects/methods
' correspond to SOAP requests and constructs the binding table
' Return True if all went wel, False if any errors
' <?xml version="1.0"?>
' <SOAPbindings>
'     <binding
'         requestMethod = ""
'         ProgId = ""
'         server = ""
'         objectMethod = ""
'     >
' </SOAPbindings>
'**********************************************************
Public Function LoadMethodBindings(sBindingFileURI As String) As Boolean
    On Error GoTo ErrHand
    
    Dim oDOM As MSXML.DOMDocument
    Dim oEl As MSXML.IXMLDOMElement
    Dim oNodeList As MSXML.IXMLDOMNodeList
    Dim sSoapMethod As String
    Dim nIdx As Integer
    
    Set oDOM = New DOMDocument
    
    If oDOM.Load(sBindingFileURI) Then
        ' Parse the file and populate the Dictionary
        ' The SOAP method name is used to build the key to retrieve
        ' the COM object information
        Set oNodeList = oDOM.getElementsByTagName("binding")
        For nIdx = 0 To oNodeList.length - 1
            Set oEl = oNodeList.Item(nIdx)
            With oEl.Attributes
                sSoapMethod = .getNamedItem("requestMethod").nodeValue
                m_dictBindings.Add sSoapMethod & DICT_PROGID_SUFFIX, _
                    .getNamedItem("progID").nodeValue
                m_dictBindings.Add sSoapMethod & DICT_METHOD_SUFFIX, _
                    .getNamedItem("objectMethod").nodeValue
                m_dictBindings.Add sSoapMethod & DICT_SERVER_SUFFIX, _
                    .getNamedItem("server").nodeValue
            End With
        Next
        LoadMethodBindings = True
    Else
        Call ClassErrors.SetErrorInfo(oDOM.parseError.reason, _
            PROGID_STRING & "LoadMethodBindings", oDOM.parseError.Errorcode)
        LoadMethodBindings = False
    End If

ErrHand:
    If Err.Number <> 0 Then
        LoadMethodBindings = False
        Call ClassErrors.SetErrorInfo(Err.Description, _
            PROGID_STRING & "LoadMethodBindings", Err.Number)
    End If
End Function
'*******************************************************************
' Public Function GetBindingInfoByName(sMethodName) As String
' Method added for debugginh purposes to check that binding information
' is loaded correctly.
'*******************************************************************
Public Function GetBindingInfoByName(sMethodName) As String
    On Error GoTo ErrHand
   
    Dim s As String
    s = ""
    s = s & "SoapMethod = " & sMethodName & vbCrLf
    s = s & "ProgID = " & m_dictBindings(sMethodName & DICT_PROGID_SUFFIX) & vbCrLf
    s = s & "Method = " & m_dictBindings(sMethodName & DICT_METHOD_SUFFIX) & vbCrLf
    s = s & "Server = " & m_dictBindings(sMethodName & DICT_SERVER_SUFFIX)
     
ErrHand:
    If Err.Number <> 0 Then
        s = ""
        Call ClassErrors.SetErrorInfo(Err.Description, PROGID_STRING & "GetBindingInfoByName", Err.Number)
    End If
    GetBindingInfoByName = s

End Function
'**********************************************************
' Public Function GetSoapBody(sPostData As String) As String
' Retrieves the XML from the http post
'**********************************************************
Public Function GetSoapBody(sPostData As String) As String
    On Error GoTo ErrHand
    Dim asPost() As String
    Dim sXML As String
    ' The data should contain an empty line; actually,
    ' two consecutive Chr(10). But it might not.
    sPostData = Replace(sPostData, vbCrLf, Chr(10))
    asPost = Split(sPostData, Chr(10) & Chr(10), 2)
    If UBound(asPost) > 0 Then
        sXML = asPost(1)
    Else
        Call ClassErrors.SetErrorInfo("Can't find POST data", _
                PROGID_STRING & "GetSoapBody", _
                -1)
        sXML = ""
    End If
ErrHand:
    If Err.Number <> 0 Then
        Call ClassErrors.SetErrorInfo(Err.Description, _
                PROGID_STRING & "GetSoapBody", _
                Err.Number)
        sXML = ""
    End If
    GetSoapBody = sXML
End Function
Public Function RetrieveMethodNamespace(sXML As String) As String
    On Error GoTo ErrHand
    
    Dim oDOM As DOMDocument
    Dim sNsURI As String
    Dim asTemp() As String
    Dim oEl As IXMLDOMElement
    Dim sMethod As String
    Dim sNsPrefix As String
    
    Set oDOM = New DOMDocument
    
    If oDOM.loadXML(sXML) Then
        Set oEl = oDOM.getElementsByTagName("SOAP:Body").Item(0)
        If (oEl Is Nothing) Then
            sNsURI = ""
            Call ClassErrors.SetErrorInfo("Can't find Body element", _
                PROGID_STRING & "RetrieveMethodNamespace", _
                -1)
        Else ' Get namspace URI
            Set oEl = oEl.childNodes.Item(0)
            sMethod = oEl.nodeName
            If InStr(sMethod, ":") Then
                asTemp = Split(sMethod, ":")
                sNsPrefix = asTemp(0)
                ' We have the namespace prefix from the tag
                ' Get the namespace attr value
                sNsURI = oEl.Attributes.getNamedItem("xmlns:" & sNsPrefix).nodeValue
            Else ' Can't parse namespace
                sNsURI = ""
                Call ClassErrors.SetErrorInfo("Can't parse namespace", _
                    PROGID_STRING & "RetrieveMethodNamespace", _
                    -1)
            End If
        End If
    Else ' Can't parse body XML
        sNsURI = ""
        Call ClassErrors.SetErrorInfo("Can't parse XML", _
                PROGID_STRING & "RetrieveMethodNamespace", _
                -1)
    End If
    
ErrHand:
    If Err.Number <> 0 Then
        Call ClassErrors.SetErrorInfo(Err.Description, _
                PROGID_STRING & "RetrieveMethodNamespace", _
                Err.Number)
        sNsURI = ""
    End If
    
    RetrieveMethodNamespace = sNsURI
    Set oDOM = Nothing
End Function
'**********************************************************
' Private Function RetrieveSoapMethodName(sXML As String) As String
' Parses out the name of the request method from the SOAP payload.
' Returns an empty string if there's an error
'**********************************************************
Public Function RetrieveSoapMethodName(sXML As String) As String
    On Error GoTo ErrHand
    
    Dim oDOM As DOMDocument
    Dim sMethod As String
    Dim asTemp() As String
    Dim oEl As IXMLDOMElement
    
    Set oDOM = New DOMDocument
    
    If oDOM.loadXML(sXML) Then
        Set oEl = oDOM.getElementsByTagName("SOAP:Body").Item(0)
        
        If Not oEl Is Nothing Then
            sMethod = oEl.childNodes.Item(0).nodeName
            If InStr(sMethod, ":") Then
                asTemp = Split(sMethod, ":")
                sMethod = asTemp(1)
            Else
                Call ClassErrors.SetErrorInfo("Could not retrive method name", _
                    PROGID_STRING & "RetrieveSoapMethodName", _
                    -1)
                sMethod = ""
            End If
        Else
            sMethod = ""
        End If
    Else
        sMethod = ""
    End If
    
ErrHand:
    If Err.Number <> 0 Then
        Call ClassErrors.SetErrorInfo(Err.Description, _
                PROGID_STRING & "RetrieveSoapMethodName", _
                Err.Number)
        sMethod = ""
    End If
    RetrieveSoapMethodName = sMethod
    Set oDOM = Nothing
End Function
'**********************************************************
' Public Sub SetDetailInfo(sDataType As String,
'                        sNamespace As String,
'                        sDesc As String,
'                        sNsURI As String,
'                        sLocation As String,
'                        lErrorcode As Long)
' Populate the detail structure
'**********************************************************
Public Sub SetDetailInfo(sDataType As String, _
                        sNamespace As String, _
                        sDesc As String, _
                        sNsURI As String, _
                        sLocation As String, _
                        lErrorcode As Long)
    With Detail
        
        .Datatype = sDataType
        .Namespace = sNamespace
        .Description = sDesc
        .NamespaceURI = sNsURI
        .Location = sLocation
        .Errorcode = lErrorcode
    End With

End Sub

'**********************************************************
' Private Sub SetSoapFault(FaultCode As SOAP_FAULT_CODE,
'                   RunCode As SOAP_RUN_CODE)
' Populate our DOAP fault member variables
'**********************************************************
Private Sub SetSoapFault(FaultCode As SOAP_FAULT_CODE, RunCode As SOAP_RUN_CODE)
   
    m_nSoapFaultCode = FaultCode
    m_nSoapRunCode = RunCode
    Select Case FaultCode
        Case Version_Mismatch:
            m_sSoapFaultDescription = _
                "The call was using an unsupported SOAP version."
            m_sSoapFaultString = "Version Mismatch"
        Case Must_Understand:
            m_sSoapFaultDescription = _
                "An XML element was received that contained an element  tagged with mustUnderstand='1' that was not understood  by the receiver."
            m_sSoapFaultString = "Must_Understand"
        Case Invalid_Request:
            m_sSoapFaultDescription = "The receiving application did not process the request because it was incorrectly formed or not supported by the application."
            m_sSoapFaultString = "Invalid Request"
        Case Application_Faulted:
            m_sSoapFaultDescription = "The receiving application faulted when processing the request. The 'detail' element contains the application-specific fault."
            m_sSoapFaultString = "Application Faulted"
        Case Else:
            m_sSoapFaultDescription = "Uknown."
            m_sSoapFaultString = "Uknown."
    End Select
End Sub

'**********************************************************
' Public Function VerifySoapMethodHeader(sSOAPPayload) As Boolean
' If we receive the HTTP header SOAPMethodName (and we *should*) then
' we need to see that the method name in the payload, and its namespace,
' match.
' Should look something like this:
' SOAPMethodName: http://electrocommerce.org/abc#MyMethod
'**********************************************************
Public Function VerifySoapMethodHeader(sPostData As String) As Boolean
    On Error GoTo ErrHand
    Dim sPayload As String
    Dim nStart As Integer
    Dim nEnd As Integer
    Dim sHeaderValue As String
    Dim asHeaderData() As String
    Dim bOK As Boolean
    
    If InStr(1, sPostData, "SOAPMethodName") Then
        nStart = InStr(1, sPostData, "SOAPMethodName:")
        nStart = nStart + Len("SOAPMethodName:") + 1
        nEnd = InStr(nStart, sPostData, Chr(10))
        sHeaderValue = Trim(Mid(sPostData, nStart, (nEnd - nStart)))
        asHeaderData = Split(sHeaderValue, "#")
        sPayload = Me.GetSoapBody(sPostData)
        If (asHeaderData(1) = RetrieveSoapMethodName(sPayload)) Then
            bOK = True
        Else
            bOK = False
        End If
    Else
        ' No header as required!
        bOK = False
    End If
    
    If Not bOK Then
        Call SetSoapFault(Invalid_Request, No)
    End If
    
ErrHand:
    If Err.Number <> 0 Then
        bOK = False
        Call SetSoapFault(Invalid_Request, No)
        Call ClassErrors.SetErrorInfo(Err.Description, _
                PROGID_STRING & "VerifySoapMethodHeader", _
                Err.Number)
    End If
    VerifySoapMethodHeader = bOK
End Function

'**********************************************************
' Public Function VerifySoapVersion(sSOAPPayload As String) As Boolean
' Check that the request is using a version of SOAP we can parse
' Return True if we can, False otherwise.
'**********************************************************
Public Function VerifySoapVersion(sSOAPPayload As String) As Boolean
    On Error GoTo ErrHand
    
    Dim oDOM As DOMDocument
    Dim oEl As IXMLDOMElement
    Dim sNS As String
    Dim bCanHandle As Boolean
    
    bCanHandle = False
    
    Set oDOM = New DOMDocument
    If oDOM.loadXML(sSOAPPayload) Then
        ' Get the namespace value from the Envelope element
        ' Should look like this:
        ' <SOAP:Envelope xmlns:SOAP="urn:schemas-xmlsoap-org:soap.v1">
        Set oEl = oDOM.childNodes(0)
        If Not oEl Is Nothing Then
            sNS = oEl.Attributes.getNamedItem("xmlns:SOAP").nodeValue
            If InStr(sNS, "soap.v1") Then
                bCanHandle = True
            Else
                bCanHandle = False
            End If
        Else
            bCanHandle = False
        End If
    Else
        bCanHandle = False
    End If
    
ErrHand:
    If Err.Number <> 0 Then
        Call ClassErrors.SetErrorInfo(Err.Description, _
                PROGID_STRING & "VerifySoapVersion", _
                Err.Number)
        bCanHandle = False
    End If
    VerifySoapVersion = bCanHandle
End Function

Private Sub Class_Initialize()
    Set m_dictBindings = New Dictionary
    Detail.Errorcode = 0
    Detail.Description = ""
    Detail.Location = ""
    Set ClassErrors = New ErrorUtils.ErrorInfo
    Call ClassErrors.ClearErrorInfo
End Sub


